pro lucy_guess,n_iter,f,psf,new,guess=guess,conv_guess=conv_guess
;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;+
;
;*NAME: LUCY_GUESS
;
;*PURPOSE: Richardson-Lucy alg. for deconvolving spectra using Bayesian methods.
;               Requires the point spread function.
;
;*CALLING SEQUENCE:
;       LUCY_GUESS, n_iter, f, psf, new, guess=guess, conv_guess=conv_guess
;
;*PARAMETERS:
;  INPUT:
;       n_iter  - (REQ) - (0)   - (I,L)     - Number of iterations.
;       f       - (REQ) - (1,2) - (I,R,L,D) - Measured flux vector.
;       psf     - (REQ) - (1,2) - (I,R,L,D) - Point spread function,
;                                             binned to 1 diode resolution.
;  OPTIONAL INPUT PARAMETER:
;       guess      - (1,2) - (I,R,L,D) - Guess of form of the deconvolved
;                                             spectrum.
;       conv_guess  - (logical) - Flag to convol the optional paramter GUESS.
;                               Note: GUESS is required to use CONV_GUESS.
;  OUTPUT:
;       NEW     - (REQ) - (1)   - (I,R,L,D) - Deconvolved spectrum.
;
;*SUBROUTINES CALLED:
;       CONVOL
;*EXAMPLES:
;
;*RESTRICTIONS:
;       Requires the point spread function to be computed previously.
;*MODIFICATION HISTORY:
;       Ver 1.0 - 01/10/90 - S. Shore     - GSFC
;       Ver 1.1 - 10/19/90 - S. Shore     - GSFC - Reverses PSF to conform with
;                                                  IDL convolution procedure.
;       Ver 2.0 - 12/15/90 - J. Blackwell - GSFC - Modified to conform with
;                                                  GHRS DAF standards.
;       1/23/92 JKF/ACC         - moved to IDL V2.
;       4/9/92  RDR   Version 2.1       - revised I/O 
;       4/9/92  JKF/ACC                 - added optional parameters
;       Ver 3.0 - 4/27/92 -  SNS        - GSFC - Modified reversal of psf
;                                                  for calculating weight.
;-
;-------------------------------------------------------------------------------
on_error,2

if n_params(0) eq 0 then begin
  message,' Calling Sequence:  LUCY_GUESS, n_iter, f, psf, NEW' ,/cont
  message,' Optional keywords: guess=guess, /conv_guess
end else begin
  zparcheck,'LUCY_GUESS',n_iter,1,[2,3],0,"n_iter invalid"
  zparcheck,'LUCY_GUESS',f,2,[2,3,4,5],[1,2],"f invalid"
  zparcheck,'LUCY_GUESS',psf,3,[2,3,4,5],[1,2],"psf invalid"
end

print,' Number of FLUX datapoints: ',n_elements(f)                      
data       = f                          ; Make copy of input data 
;psf       = reverse(psf)               ; flip the PSF.
psfr       = reverse(psf)               ; flip the PSF. (SNS 4/27/92)
;
; Did user specified initial guess? If so, did user specify that the
;   initial guess should be convolved?
;
if keyword_set(guess) then begin
        if keyword_set(conv_guess) then begin
                conv_guess=convol(guess,psfr)
                ic=where(conv_guess ne 0, found)
                if found gt 0 then begin
                        imin=min(ic)
                        imax=max(ic)
                        print,' Convolved GUESS     min: ',imin,'  max:',imax
                        g0=conv_guess(imin)
                        g1=conv_guess(imax)
                        for k=0,imin-1 do conv_guess(k)=g0 
                        for kk=imax+1,n_elements(f)-1 do conv_guess(kk)=g1  
                end else $
                        message,' Unable to convolve GUESS (All ZEROES)'

        end else conv_guess = guess
        old=conv_guess
end else begin
        conv_guess =fltarr(n_elements(f))
        old=data
end                                                              
;
;  Iterate 
;
for k=0,n_iter do begin
    weight = convol(old,psfr)
    new    = old * convol( (data/weight) ,psf)
    old    = new
endfor

return ; LUCY_GUESS
end ; LUCY_GUESS
