;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;!                                                                             !
;! This routine belongs to the PHT Interactive Analysis Package PIA.           !
;!                                                                             !
;! PIA is a joint development by the ESA Astrophysics Division and the ISOPHOT !
;! Consortium led by the Max Planck Institute for Astronomy (MPIA), Heidelberg.! 
;! Contributing ISOPHOT Consortium institutes are DIAS, RAL, AIP, MPIK, and    !
;! MPIA.                                                                       !
;!                                                                             ! 
;! Its use and distribution are free, as long as this header text is not       !
;! removed, or changed. When publishing ISOPHOT Data reduced with this analysis!
;! package, please mention this in the acknowledgement.                        !
;!                                                                             !
;! No warranties for installation/ support/ maintenance are given.             ! 
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;+
; NAME:
;       PIA_PROC_GEN
;
; PURPOSE:
;       Menu and procedure within the PIA to get automatically
;       a sequence of steps to be performed, without major interaction.
;       Data reduction from ERD up to AAP level, including calibration.
;
; CATEGORY:
;	PIA - processing
;
; CALLING SEQUENCE:
;       PIA_PROC_GEN
;
; INPUTS:
;	none
;
; KEYWORD PARAMETERS:
;	SRD:		If present, processing stopped at SRD level
;	SCP:		If present, processing stopped at SCP level
;	FROM_SRD:	If present, processing started at SRD level
;	FROM_SCP:	If present, processing started at SCP level
;	ABS_PHT:	Absolute photometry ('dark'+'straylight' recognition)
;	SPARSE_MAP:	Sparse map case
;	PER_RPID:	Reduce long measurement dividing per RPID
;	AUTO_REC:	Recognize for AOT type and reduce accordingly
;	GROUP:	The group leader of this widget hierarchy
;
; OUTPUTS:
;	none
;
; SIDE EFFECTS:
;	If the FITS files are chosen as input, the ERD dynamic structure
;	is expanded with all the measurements contained in those files.
;	The dynamic structures containing the SRD, SCP, SPD and AAP levels
;	are expanded with all the measurements to be processed.
;
; COMMON BLOCKS:
;	phterd:		ERD dynamic structure
;	phtsrd:		SRD dynamic structure
;	phtscp:		SCP dynamic structure
;	phtspd:		SPD dynamic structure
;	phtaap:		AAP dynamic structure
;	phtresact:	actual responsivities 
;	deglitch_param: parameters for deglitching procedure
;	ibcksub_param:	background subtraction parameters
;	gen_choices:	parameters for AOT/Batch procedures driving
;	splitramp_par:	ramp subdivision parameters
;	widget_log_cmn:	info window common
;	phtpatt:	pattern analysis parameters
;
; PROCEDURES USED:
;	PIA Procedure(s):
;		deglitch_param_init
;		lin_voltages
;		pia_gencho
;		pia_load_ierd
;		pia_load_iscp
;		pia_load_isrd
;		pia_pickfile
;		pia_respons
;		pia_rodisc
;		pia_sigdisc
;		pickmeas
;		process_erd
;		process_scp
;		process_srd
;		read_erd
;		respons_interpol
;	Internal:
;		pia_proc_gen_event
;
; MODIFICATION HISTORY:
;       Written by: 	Carlos Gabriel (ESA/ESTEC)		January 1995
;	Last Modification:	
;		P05/P25 case extended & saves also,	
;		new use of gen_choices (CG) 			April 1997
;	V6.2
;	Batch and per RPID modes integrated, 
;	 correction for reset interval and vignetting and
;	 ramps sub-division capability added  (CG)		May 1997
;	Still some improvements to provide more info in the 
;	log screen			(JAcosta)		June 1997
;	V6.3 
;	Using subtr_darstr_meas extension +
;	 correct treatment of RPID division +
;	 extension in responsivity choices (CG)			June 1997
;	V6.4
;	Using decoded administration names & 
;	 Stop at SCP if per raster point (CG)			September 1997
;	V6.5
;	call to add_coordinates changed (JAc)			 February 1998
;	FCS limits check -> def.Resp. if OoL (CG)		 February 1998
;	V7.0
;	Problem with FCSN=-1 fixed (CG)				 February 1998
;	Def. Resp. also for interpolation if OoL (CG)		 March 1998
;	2 Threshold deglitching added (CG)			 March 1998
;	Default responsivity choice corrected for sequences
;	 with more than 2 calibration measurements (CG)		March 1998
;	V7.1 
;	Respons choices extended,
;	 bug for SPD FITS file header fixed (JA+CG)		July 1998
;	V7.2
;	Signal linearization included (CG)
;	Pattern processing included (CG)			March 1999
;	V7.4
;	Bug fixed: header by deglitching overwritten (CG)	July 1999
;	V7.9
;	Vignetting correction not performed if pattern (CG)	July 1999
;	+ logic changed for including corrections on SRD
;	  level if pattern and corrections on SCP required
;	Extension of "pattern proc" to sawtooth (CG)		July 1999
;	Setting RPID>[1,1] for tracking at ERD instof SRD (CG)	September 1999
;	V8.0
;	Adding PHT-S pointing correction (CG)			December 1999
;	V8.2
;	Bug for writing PHT-S AAP from dyn-response fixed (CG)  July 2000
;	V9.0
;-

PRO PIA_PROC_GEN, SRD=SRD, SCP=SCP, FROM_SRD=FROM_SRD, FROM_SCP=FROM_SCP, $
	GROUP=GROUP, ABS_PHOT=ABS_PHOT, PER_RPID=PER_RPID, AUTO_REC=AUTO_REC, $
	SPARSE_MAP=SPARSE_MAP, SUMM_INFO=summ_info

@phtacc
@phtcs
@phtdrift
@phterd
@phtsrd
@phtscp
@phtspd
@phtaap
@phtresact
;@deglitch_param
@ibcksub_param
@gen_choices
@splitramp_par
COMMON widget_log_cmn, wd_text, wd_title
COMMON phtpatt, patt_str

 	title = 'General AOT / Batch Processing '
	IF KEYWORD_SET(PER_RPID) THEN $
	 title = 'Processing of a VERY LONG raster measurement'


	wd_title = title
	WIDGET_LOG, wd_base, ACTION='CREATE', XSIZE=75

;
;  Get the necessary information to start
;
	
	IF N_ELEMENTS(gen_respalt) EQ 0 THEN gen_choices_init
	IF KEYWORD_SET(from_srd) OR KEYWORD_SET(from_scp) THEN $
	   gen_fits.FITS = 10
	   
        pia_gencho, group=wd_base

	IF N_ELEMENTS(gen_filt) EQ 1 THEN BEGIN 
		CASE STRMID(gen_filt,0,2) OF 
			'PP': dtyp = 'P#'
			'P1': dtyp = 'C1'
			'P2': dtyp = 'C2'
			'PS': dtyp = 'SL'
			ELSE: dtyp = ''
		ENDCASE
	ENDIF	

        IF gen_stop EQ 1 THEN BEGIN
                MESSAGE, /INFO, ' AOT Procedure stopped'
		WIDGET_LOG, wd_base, ACTION='DESTROY' 
                RETURN
        ENDIF


;
; ERD Selection criteria 
; 
	IF (gen_select.erdsel) THEN pia_rodisc, nndr = -1

;
; SRD Selection criteria 
; 
	IF (gen_select.srdsel) THEN pia_sigdisc, nsig = -1


; Corrections menues:
	pia_corrections, erd=gen_correct.erd_selcor, $
	   srd=gen_correct.srd_selcor, scp=gen_correct.scp_selcor, $
	   group=wd_base


;
; Select ERD Files or individual measurements using multiselectors
;


; FITS files
	IF gen_fits.FITS EQ 0 THEN BEGIN

	  IF KEYWORD_SET(PER_RPID) THEN BEGIN
	   PRINT, 'Only a measurement already in buffer will be accepted'
	   PRINT, 'Returning ...'
	   RETURN
	  ENDIF



	  fillist = pia_pickfile(/multi, filter=gen_filt, path=!FITDIR, $
	  		get_path=newpath, group=wd_base)
	  IF newpath NE '' THEN !FITDIR=newpath 

          IF fillist(0) EQ '' THEN BEGIN
                MESSAGE, /INFO, $
		  ' No Data selected, AOT Procedure Sequence stopped'
		WIDGET_LOG, wd_base, ACTION='DESTROY'
                RETURN
          ENDIF


;
; Compute first measurement to be used
; 
          firstscp = N_ELEMENTS(phtscp)
;

; Read ERD files	
;	 					
;	  FOR I=0,N_ELEMENTS(fillist)-1 DO BEGIN 
;
;        	filename = STRUPCASE(STRTRIM(fillist(i),2))
;        	filename = strmid(filename,(STRPOS(filename,'.FITS') - 12), 17)
;
;        	WIDGET_CONTROL, text1, SET_VALUE='Reading ERD '+filename, $
;			/append
;		WIDGET_CONTROL, base, /hourglass
;
;		read_erd, filename,hdr,ilunr
          	
; Compute first measurement on the ERD level
;		IF I EQ 0 THEN firsterd = ilunr(0)

;	  ENDFOR		

; If first measurement of each file to be displayed		
;	  pia_disperd, phterd(ilunr(0))

;
; Compute last measurement to be used
; 
;          lasterd = max(ilunr)

;
; Compute all measurements to be processed to SRD 
;

;	  meas = indgen(lasterd-firsterd+1) + firsterd

	ENDIF ELSE BEGIN

; individual measurements

	  IF KEYWORD_SET(from_scp) THEN BEGIN 
	    measnames = pickmeas(FTYPE="SCP", DTYPE=dtyp, /multi, posi=posi,$
	    		GROUP=wd_base)

            IF measnames(0) EQ '' THEN BEGIN
                MESSAGE, /INFO, ' No Data selected, AOT procedure stopped'
		WIDGET_LOG, wd_base, ACTION='DESTROY'
                RETURN
            ENDIF

	    WIDGET_LOG, wd_base, TEXT = 'Starting from SCP level  ',$ 
	        ACTION='APPEND'

	    meas = 0L

              FOR I=0,N_ELEMENTS(measnames) - 1 DO BEGIN
	  	measadd = WHERE(phtscp.admn EQ measnames(I),n)
		meas = [meas,measadd(posi(i))]
	      ENDFOR

	    scpps = meas(1:*)
	    firstscp = scpps(0)

	  ENDIF ELSE IF KEYWORD_SET(from_srd) THEN BEGIN 
	    measnames = pickmeas(FTYPE="SRD", DTYPE=dtyp, /multi, posi=posi,$
	    		GROUP=wd_base)

            IF measnames(0) EQ '' THEN BEGIN
                MESSAGE, /INFO, ' No Data selected, AOT procedure stopped'
		WIDGET_LOG, wd_base, ACTION='DESTROY'
                RETURN
            ENDIF

            WIDGET_LOG, wd_base, TEXT ='Starting from SRD level  ', $
	        ACTION='APPEND'

	    meas = 0L

              FOR I=0,N_ELEMENTS(measnames) - 1 DO BEGIN
	  	measadd = WHERE(phtsrd.admn EQ measnames(I),n)
		meas = [meas,measadd(posi(i))]
	      ENDFOR
	    srdps = meas(1:*)
	    
	  ENDIF ELSE IF KEYWORD_SET(per_rpid) THEN BEGIN

	    measname = pickmeas(FTYPE="ERD", DTYPE=dtyp, posi=posi,$
	    		GROUP=wd_base)
	    meas = WHERE(phterd.admn EQ measname(0),n)

            IF measname(0) EQ '' THEN BEGIN
                MESSAGE, /INFO, ' No Data selected, AOT procedure stopped'
		WIDGET_LOG, wd_base, ACTION='DESTROY'
                RETURN
            ENDIF
	    
	    ;;IF n GT 1 THEN meas = meas(posi(0))

	    pia_load_ierd, phterd(meas(posi(0))), tmperd
	    firsterd = N_ELEMENTS(phterd)
	    sep_rp, tmperd 
	    lasterd = N_ELEMENTS(phterd) - 1

	    meas = indgen(lasterd-firsterd+1) + firsterd	    
	    
            WIDGET_LOG, wd_base, TEXT ='Starting with measurement  ', $
	        ACTION='APPEND'

	  ENDIF ELSE BEGIN

	    measnames = pickmeas(FTYPE="ERD", DTYPE=dtyp, /multi, posi=posi,$
	    		GROUP=wd_base)

            IF measnames(0) EQ '' THEN BEGIN
                MESSAGE, /INFO, ' No Data selected, AOT procedure stopped'
		WIDGET_LOG, wd_base, ACTION='DESTROY'
               RETURN
            ENDIF

            WIDGET_LOG, wd_base, TEXT ='Starting with measurements  ', $
	        ACTION='APPEND'

	    meas = 0L
	    
              FOR I=0,N_ELEMENTS(measnames) - 1 DO BEGIN
	  	measadd = WHERE(phterd.admn EQ measnames(I),n)
	    	;;IF n GT 1 THEN measadd = measadd(posi(i))
		meas = [meas,measadd(posi(i))]
	      ENDFOR
	    meas = meas(1:*)
	  ENDELSE	
; 
          firstscp = N_ELEMENTS(phtscp)

	  fillist = 0
	ENDELSE



; Processing starts here
 
	IF KEYWORD_SET(from_scp) THEN GOTO, SCP_START

	IF KEYWORD_SET(from_srd) THEN GOTO, SRD_START

; Read ERD files	
	 					
	FOR IFITS=0,N_ELEMENTS(fillist)-1 DO BEGIN 
	
	  IF gen_fits.FITS EQ 0 THEN BEGIN
	   
        	filename = STRUPCASE(STRTRIM(fillist(ifits),2))
        	filename = strmid(filename,(STRPOS(filename,'.FITS') - 12), 17)

        	WIDGET_LOG, wd_base, TEXT ='Reading ERD '+filename, $
	          ACTION='APPEND'

		read_erd, filename,hdr,ilunr, path=newpath
          	
; Compute first measurement on the ERD level
;		firsterd = ilunr(0)

	  	meas = ilunr
	  ENDIF

;
; Process all data to srd, including ERD corrections
;
        WIDGET_LOG, wd_base, TEXT =' Processing from ERD->SRD:', $
	             ACTION='APPEND'
	
        WIDGET_LOG, wd_base, TEXT =' (This could take some time, take it easy) ', $
 	             ACTION='APPEND'


	IF gen_process.splitrp THEN BEGIN 
	  div_ramp=1 
          WIDGET_LOG, wd_base, TEXT =' Ramp subdivision will be performed ', $
	             ACTION='APPEND'
          WIDGET_LOG, wd_base, $ 
                     TEXT =' using last chosen values (or default ones)  ', $
	             ACTION='APPEND'
        ENDIF ELSE div_ramp=0
	 
; Open loop over index meas
	FOR i=0, N_ELEMENTS(meas)-1 DO BEGIN
             pia_load_ierd, phterd(meas(i)), tmperd
	     newhdr = tmperd.hdr
	     
; Decode pixf, returns DR and OTF
	 decode_pixf,  tmperd.pixf, susp, chpon, chps, dr, otf    
; Look into the selection criteria
        get_rodisc, tmperd, /ENCODE_FLAG, DR=dr , OTF=otf 

; Flag also the saturated voltages (only for real pixels)
	get_accvolt, tmperd, dr, /ENCODE_FLAG

; Get valid flag concerning saturation and selection criteria
	get_validflag, tmperd.flag, 'ERD',redflag, $
		   det= tmperd.det,/allpix
		      
; get index with the ramp ends
	GIVE_cs, tmperd, cst, record, conv_cst
        GET_RAMPEND, tmperd.time, dr, tmperd.tint, rampflag, rampend, $
		DATRED=cst.drs(record)
		   
;
;Setting all RPIDs to [1,1] for solar tracking observations
;
	IF (gen_correct.rpid0) THEN BEGIN

	  ATT = FXPAR(newhdr, 'ATTTYPE')
	  IF STRTRIM(ATT) EQ 'T' THEN BEGIN
        	WIDGET_LOG, wd_base, TEXT =' Solar tracking obs detected ', $
	             ACTION='APPEND'
        	WIDGET_LOG, wd_base, TEXT =' -> setting RPIDs to [1,1] ',   $
	             ACTION='APPEND'

	  	tmperd.rpid = tmperd.rpid * 0 + 1b
		FXADDPAR, newhdr, 'PR_RPID0', 'RPID change', $
		       "All RPID's set to [1b,1b] ", BEFORE='XTENSION'
	  ENDIF
	  
	ENDIF
	
; Linearization
;
	  IF (gen_correct.linear) THEN BEGIN
                WIDGET_LOG, wd_base, TEXT =' Linearizing ramps for ' + $
		 dec_admn(tmperd.admn), ACTION='APPEND'
		lin_done = fxpar(newhdr,'PR_LINC',count=count)
		IF count EQ 0 THEN $
		  lin_voltages, tmperd, newhdr, /silent $
		ELSE $
          	  WIDGET_LOG, wd_base,  $
            	     TEXT=' Linearization already performed, skipped', $
	             ACTION='APPEND'
	  ENDIF

;
; ERD-deglitching
;
	  IF (gen_correct.deglit2) THEN BEGIN
                WIDGET_LOG, wd_base, TEXT =' Deglitching ramps for '+ $
		    dec_admn(tmperd.admn), ACTION='APPEND'
		ramp_deglitch, tmperd, newhdr, redflag, rampend, $ 
		   /silent, /encode_flag, text_out=text_out
		IF text_out(0) NE '' THEN $
                 WIDGET_LOG, wd_base, TEXT = text_out, ACTION='APPEND'
		
	  ENDIF
;
; ERD-deglitching - Two thresholds
;
	  IF (gen_correct.pairdeglit) THEN BEGIN
                WIDGET_LOG, wd_base, TEXT =' Two thresholds deglitching: '+ $
		    dec_admn(tmperd.admn), ACTION='APPEND'
		ramp_2t_deglitch, tmperd, redflag, newhdr,/ENCODE_FLAG
	  ENDIF
	
	  ;; In principle it is not necessary to save modification in
	  ;;  phterd dynamic structure
	  ; fill_erd, tmperd, newhdr, REPLACE = meas(i)
	    
;
; Process to Pattern
;
	  IF gen_process.pattern EQ 1 AND tmperd.det NE 'SL' AND $
	  	(conv_cst.cmod(record) EQ 'Rectangular' OR $
		 conv_cst.cmod(record) EQ 'Scan triangular' OR $
		  (conv_cst.cmod(record) EQ 'Scan sawtooth' AND $
		   conv_cst.cste(record) EQ '1') OR $
		 conv_cst.cmod(record) EQ 'FCS1<->FCS2') THEN BEGIN      
             WIDGET_LOG, wd_base, TEXT =' Processing ERD to Pattern: '+ $
		    dec_admn(tmperd.admn), ACTION='APPEND'
	     IF N_ELEMENTS(patt_str) EQ 0 THEN $
		    patt_str={disc_framp:0, log_ramps:8}
	     process_erd_to_pattern, tmperd, $
	  	      newhdr=newhdr, srdpos=srdps_tmp,$
	  	      disc_framp=patt_str.disc_framp, $
	  	      log_ramps=patt_str.log_ramps
;
; Normal Process ERD to srd 		      
;
	  ENDIF ELSE BEGIN                        
	     text = '    ... processing ' + dec_admn(phterd(meas(i)).admn)
	     WIDGET_LOG, wd_base, TEXT =text, ACTION='APPEND'
	     process_tmperd, tmperd, 1, newhdr, srdps_tmp, div_ramp=div_ramp, $
	        DR = dr
	  ENDELSE
	  	  
	  IF N_ELEMENTS(srdps) EQ 0 THEN srdps=srdps_tmp ELSE $
	 	srdps=[srdps,srdps_tmp]

	ENDFOR  ;close loop over index meas
		

; close loop for ERD files, pseudo-loop for measurements, IFITS
; Free memory in case of sep_rp, delete all erd RP measurements
          IF gen_fits.delete THEN erd_init, INDEX=meas  $
	     ELSE IF KEYWORD_SET(PER_RPID) THEN BEGIN
               WIDGET_LOG, wd_base, $  
                TEXT ='Deleting all pseudo-measurements  ERD / RPid', $
	        ACTION='APPEND'
	    erd_init,INDEX=meas	
	  ENDIF	

       ENDFOR  ;close loop over full measurement


SRD_START:
;
;
; Loop over SRD structures

	FOR i=0,N_ELEMENTS(srdps)-1 DO BEGIN

	  pia_load_isrd, phtsrd(srdps(i)), tmpdata
	  
	  get_sigdisc, tmpdata, /ENCODE_FLAG
	  
	  newheader = tmpdata.hdr

          WIDGET_LOG, wd_base, $
           TEXT =['','Now at SRD level of '+dec_admn(tmpdata.admn)], $
        	    ACTION='APPEND'


	pat = fxpar(newheader, 'PR_PATT', count=count)
	IF count GT 0 AND gen_process.pattern THEN pattern=1 ELSE pattern=0


;
;  Reset interval correction
;
	  IF (gen_correct.resint) OR $
	     (pattern AND gen_correct.resint_scp) THEN BEGIN
	  
; check: if correction was already applied 
; apply the correction and write info into the header 
	  
	     do_corr_resetint, tmpdata, newheader, INFO = info
	  
	     WIDGET_LOG, wd_base, TEXT = info, ACTION = 'APPEND'

	  ENDIF


;
;  Deglitching
;

	  IF (gen_correct.deglit) THEN BEGIN

        	WIDGET_LOG, wd_base, TEXT = ' Deglitching ...', $
	             ACTION='APPEND'

		indsevere_deglitch_srd = set_indsevere('SRD',    $ 
		     ['NOTF','GLITCH'],/DELETE) 
		get_validflag, tmpdata.flag, 'SRD',redflag,$
		  INDSEVERE = indsevere_deglitch_srd,DET=tmpdata.det,/allpix 

		DO_DEGLITCH, tmpdata, redflag, newheader, degl_acc, $ 
		   /ENCODE_FLAG
				
	  ENDIF

;
; Subtract dark current 
;
	  IF (gen_correct.darkc_srd) OR $
	     (pattern AND gen_correct.darkc)THEN BEGIN

                WIDGET_LOG, wd_base, TEXT =' Subtracting Dark Current', $
	             ACTION='APPEND'

		dark_done = fxpar(newheader,'PRS_DARK',count=count)
	        IF KEYWORD_SET(ABS_PHOT) THEN count = 10 ELSE $
		IF KEYWORD_SET(auto_rec) THEN BEGIN
	  		AOT = STRTRIM(FXPAR(newheader, 'EOHAAOTN'),2)
	  		IF AOT EQ 'P25' or AOT EQ 'P05' THEN $
	  		   count = 10
	  	ENDIF 

		CASE count OF 

		0: darkcur_orb, tmpdata, newheader=Newheader

                10: BEGIN 
                  text = [' Dark Current subtraction with defaults required', $
                          ' but not performed: P05/P25 case']
           	  WIDGET_LOG, wd_base, TEXT =text, ACTION='APPEND'
                END

		ELSE: BEGIN
          	  WIDGET_LOG, wd_base, $ 
          	   TEXT =' Dark Current subtraction already performed, skipped',$
	             ACTION='APPEND'
		END

          	ENDCASE

          ENDIF 

;
;  Signal linearization
;
	  IF tmpdata.det NE 'SL' THEN BEGIN
	   IF (gen_correct.siglin) OR $
	     (pattern AND gen_correct.siglin2) THEN BEGIN
	  
	      rlin_proc, tmpdata, newhdr=newheader, /SRD, INFO=info
	      IF info EQ '' THEN $
	       info = ' Signal linearization performed (SRD level)'

	     WIDGET_LOG, wd_base, TEXT = info, ACTION = 'APPEND'

	   ENDIF
	  ENDIF 

;
; replace phtsrd structure by tmpdata
;
	fill_srd, tmpdata, newheader, replace = srdps(i)

; Write SRD measurements into PIA internal files
	
	IF KEYWORD_SET(per_rpid) THEN GOTO, NO_SAVE
          IF (gen_save.savsrd) THEN BEGIN
	    write_isrd, srdps(i), INFO = infosave 
            WIDGET_LOG, wd_base, TEXT =infosave, ACTION='APPEND' 
	    indcs = WHERE(phtcs.unit EQ STRMID(tmpdata.admn,4,8), count)
	    IF count GT 0 THEN BEGIN
		      PIA_LOAD_ICS, phtcs(indcs(0)), phtics
		      FILE='CS'+phtics.UNIT+'.ICS'
		      junk = FINDFILE(!intdir+FILE, COUNT=count)
		      IF count GT 0 THEN $
			MESSAGE, /INFO, $
		  	  'Compact Status file not written, already present' $
		      ELSE SAVE, phtics, FILE=!INTDIR+file, /XDR
	    ENDIF ELSE MESSAGE, /INFO, $
		    	'Compact Status not found in common, not saved' 
	  ENDIF
	  
          IF (gen_save.savsrdf) THEN BEGIN
	    write_srd, tmpdata, newheader
	    text = ["Saving measurement " + dec_admn(tmpdata.admn), $
	         " as FITS file: " + !INTDIR + tmpdata.admn + '_SRD.FITS', ''] 
            WIDGET_LOG, wd_base, TEXT =text, $
             ACTION='APPEND' 
	  ENDIF
	  
NO_SAVE:
	  ; Do not process if just to SRD
	  IF KEYWORD_SET(SRD) THEN GOTO, END_TMPSRD

          WIDGET_LOG, wd_base,  $
        	    TEXT =' Processing measurements  SRD->SCP', ACTION='APPEND'

	  pat = fxpar(newheader, 'PR_PATT', count=count)
	  IF count GT 0 AND gen_process.pattern THEN pattern=1 ELSE pattern=0
; include reset interval, dark current and linearisation
	  
; Dynamic response if PHT-S staring and required
	  IF tmpdata.det EQ 'SL' AND max(tmpdata.step) EQ 1 AND $
	  	max(tmpdata.rpid) EQ 1 AND gen_process.dynresp THEN BEGIN
; subtract dark current if necessary
		
	     dark_done = fxpar(newheader,'PRS_DARK',count=count)
	     IF not dark_done THEN BEGIN

		darkcur_orb, tmpdata, newheader=newheader
                 text = [' Dark Current subtraction mandatory ', $
                          ' for Dynamic Response, done']
           	  WIDGET_LOG, wd_base, TEXT =text, ACTION='APPEND'
	     ENDIF
	     
  	     process_srddynr, tmpdata, hdr=newheader, tmpdynr = tmpdynr  

  	     dynresp2aap, tmpdynr, aapps

             text = ['Processing using Dynamic Response ', $
                          ' measurement '+dec_admn(tmpdata.admn) ]
              WIDGET_LOG, wd_base, TEXT =text, ACTION='APPEND'

; save here if required	      
             IF (gen_save.savaap) THEN BEGIN
	        FOR iaapps=0,N_ELEMENTS(aapps)-1 DO BEGIN
	          pia_load_iaap, phtaap(aapps(iaapps)), phtiaap
            	  filename = phtiaap.admn+'.IAAP'
            	  save, phtiaap, FILE=!INTDIR+filename, /XDR
            	  WIDGET_LOG, wd_base,  $  
                    TEXT='Saving AAP measurement into '+!INTDIR+filename, $
                    ACTION='APPEND'
		ENDFOR
             ENDIF
	     IF (gen_save.savaapf) THEN BEGIN
	        filename=!INTDIR+phtiaap.admn + '_AAP.FITS'

	        text = ["Saving measurement " + dec_admn(phtiaap.admn), $
	            " as FITS file: " + filename,''] 
                WIDGET_LOG, wd_base, TEXT =text, $
                ACTION='APPEND' 

		posi = aapps
		mask = [0,intarr(17) + 1]
		write_aap_multi, posi, accvector=mask, filename=filename
	     ENDIF
	     
 
	  ENDIF ELSE $
  	   process_tmpsrd, tmpdata, newheader, scppos, $
	  	driftcorr=gen_process.drft, pattern=pattern		


END_TMPSRD:

   	ENDFOR         ; end loop for SRD 

; Stop if just to SRD
	IF KEYWORD_SET(SRD) THEN GOTO, EXIT


SCP_START: 

; last measurement in SCP
	lastscp = N_ELEMENTS(phtscp) -1

	IF NOT KEYWORD_SET(from_SCP) THEN BEGIN
	        IF lastscp LT firstscp THEN GOTO, exit 
		scpps = indgen(lastscp-firstscp+1) + firstscp
	ENDIF

	IF KEYWORD_SET(PER_RPID) THEN BEGIN

          WIDGET_LOG, wd_base,  $
        	TEXT ='Deleting all pseudo-measurements  SRD / RPid', $
        	ACTION='APPEND'
	  FOR I=0,N_ELEMENTS(srdps)-1 DO BEGIN
		FOR ihan=7,21 DO HANDLE_FREE, phtsrd(srdps(i)).(ihan)
	  ENDFOR
	  phtsrd = phtsrd(0:srdps(0)-1)


	  indices = indgen(lastscp-firstscp+1) + firstscp
          WIDGET_LOG, wd_base,  $
        	TEXT ='Merging all SCP Raster points into SCP measurement', $
        	ACTION='APPEND'
	  append_rp, indices, scpps

          WIDGET_LOG, wd_base,  $
        	TEXT ='Deleting all pseudo-measurements  SCP / RPid', $
        	ACTION='APPEND'

	  FOR I=0,N_ELEMENTS(indices)-1 DO BEGIN
		FOR ihan=5,23 DO HANDLE_FREE, phtscp(indices(i)).(ihan)
	  ENDFOR
	  phtscp = [phtscp(0:indices(0)-1), phtscp(scpps(0))]
	  scpps = indices(0)
	ENDIF
  
; Start loop over SCP

	FOR i=0,N_ELEMENTS(scpps)-1 DO BEGIN

	  pia_load_iscp, phtscp(scpps(i)), tmpscp
	  detpix = pixel_det(tmpscp.det)	   
	  
	  newheader = tmpscp.hdr

          WIDGET_LOG, wd_base, $
          	TEXT =['','Now at SCP level of '+dec_admn(tmpscp.admn)], $
        	    ACTION='APPEND'

;
;Reset interval correction at SCP level
;
	  IF (gen_correct.resint_scp) THEN BEGIN

; check: if correction was already applied 
; apply the correction and write info into the header 
	  
	     do_corr_resetint, tmpscp, newheader, INFO = info, /SCP
	  
	     WIDGET_LOG, wd_base, TEXT = info, ACTION = 'APPEND'

	  ENDIF


;
; Subtract dark current 
;
	  IF (gen_correct.darkc) THEN BEGIN

                WIDGET_LOG, wd_base, TEXT =' Subtracting Dark Current', $
	             ACTION='APPEND'

		dark_done = fxpar(newheader,'PRS_DARK',count=count)
	        IF KEYWORD_SET(ABS_PHOT) THEN count = 10 ELSE $
		IF KEYWORD_SET(auto_rec) THEN BEGIN
	  		AOT = STRTRIM(FXPAR(newheader, 'EOHAAOTN'),2)
	  		IF AOT EQ 'P25' or AOT EQ 'P05' THEN $
	  		   count = 10
	  	ENDIF 

		CASE count OF 

		0: darkcur_orb, tmpscp, newheader=Newheader, /SCP

                10: BEGIN 
                  text = [' Dark Current subtraction with defaults required', $
                          ' but not performed: P05/P25 case']
           	  WIDGET_LOG, wd_base, TEXT =text, ACTION='APPEND'
                END

		ELSE: BEGIN
          	  WIDGET_LOG, wd_base, $ 
          	   TEXT =' Dark Current subtraction already performed, skipped',$
	             ACTION='APPEND'
		END

          	ENDCASE

          ENDIF 

;
;  Signal linearization SCP
;
	  IF tmpscp.det NE 'SL' THEN BEGIN
	   IF (gen_correct.siglin2) THEN BEGIN
	  
	      rlin_proc, tmpscp, newhdr=newheader, INFO=info
	      IF info EQ '' THEN $
	       info = ' Signal linearization performed (SCP level)'
	       
	     WIDGET_LOG, wd_base, TEXT = info, ACTION = 'APPEND'

	   ENDIF
	  ENDIF
;
; Perform vignetting correction
;

	  IF (gen_correct.vignet) THEN BEGIN
               WIDGET_LOG, wd_base, TEXT =' Vignetting correction ...', $
	             ACTION='APPEND'

	        pat = fxpar(newheader, 'PRS_PATT', count=count)
	        

		done = fxpar(newheader,'PRC_VIGN')
		IF done THEN BEGIN
          	  WIDGET_LOG, wd_base, $ 
          	   TEXT =' Vignetting correction already performed, skipped', $
	             ACTION='APPEND'
		ENDIF ELSE IF pat THEN BEGIN
          	  WIDGET_LOG, wd_base, $ 
          	   TEXT =' Vignetting corr. included in "pattern", skipped', $
	             ACTION='APPEND'
		ENDIF ELSE $
		  PIA_CORR_VIGN, tmpscp, newheader
		 
	  ENDIF	

;
; Perform background subtraction
;

	  IF (gen_correct.bcksub) THEN BEGIN
	  
	    IF N_ELEMENTS(ibcksub_params) EQ 0 THEN ibcksub_param_init
	    
	    srcstep = ibcksub_params.srcstep
	    bckstep = ibcksub_params.bckstep

	    ; no FCS required
	    inda = WHERE(tmpscp.fcsn NE 0, nrfcs)
            decode_admn, tmpscp.admn, decstr
	    IF nrfcs EQ 0 THEN BEGIN
	      max_step = MAX(tmpscp.step)	
	      IF max_step GT 1 AND max_step LE 4 THEN BEGIN		
		 subtr_bckg, tmpscp, srcstep, srcsubt, BCKSTEP=bckstep
		 WRITE_SUBTR, srcsubt, tmpscp, 0, newheader, OUTMEAS=tmpsub,  $
		  /NO_INFO, REDUCE_MED = ibcksub_params.use_median,           $
		  /NO_INTO_BUFFER 
;   replace structure index for further processing & temporary by subtracted
		 tmpscp = tmpsub
            	 WIDGET_LOG, wd_base, $ 
          	   TEXT =' Performing background subtraction ', $
	             ACTION='APPEND'
	      ENDIF 
	     ENDIF
	   
	   
	   ENDIF

; SL chopped dynamic calibration

	  slchop_proc = 0
	  IF (gen_process.slchop) AND tmpscp.det EQ 'SL' THEN BEGIN
	     pia_subt = FXPAR(newheader, 'PRC_SUBT', count=count)
	     max_step = MAX(tmpscp.step)	
	     
	     IF count GT 0 THEN BEGIN
	        first_corr = 1  ; it should be possible to customize 
		extrapolate = 1 ; it should be possible to customize 
	        sig_subt = FLTARR(128,2)
		sig_subt(*,0) = tmpscp.aver(0:127)
		sig_subt(*,1) = tmpscp.medi(0:127)

		give_cs, tmpscp, cst, in, conv_cst, error=error

		IF not ERROR THEN $
		 t_onchop = FLOAT(conv_cst.ndr(in)) * FLOAT(conv_cst.intt(in)) $
		ELSE t_onchop = 0. 

		calib_choppmeas_sl, sig_subt, flux_subt, flag, sprf_act, $
		 t_onchop, FIRST_CORR=first_corr, EXTRAPOLATE=extrapolate, $
		    INFO=info_slchop
		flave = flux_subt(*,0)
		flmed = flux_subt(*,1)
		flunc = tmpscp.avun(0:127)/sprf_act(*,0)
		flq1  = tmpscp.q1me(0:127)/sprf_act(*,1)
		flq3  = tmpscp.q3me(0:127)/sprf_act(*,1)
		fluxstr = {flave:flave,flunc:flunc, flmed:flmed, $
		   flq1:flq1,flq3:flq3,flflag:flag(*,0)}    

		IF N_ELEMENTS(info_slchop) NE 0 THEN $   
            	 WIDGET_LOG, wd_base, $ 
          	   TEXT =info_slchop, $
	             ACTION='APPEND'

	        scp2aap_sl, tmpscp, fluxstr, aapps, NEWHDR=newheader,$
		  INFO = info
          	WIDGET_LOG, wd_base, $ 
          	   TEXT =' Chopped PHT-S dynamic calibration performed', $
	             ACTION='APPEND'
	        
		slchop_proc = 1
		IF N_ELEMENTS(scpps_slch) EQ 0 THEN scpps_slch = scpps(i) $
		ELSE scpps_slch = [scpps_slch, scpps(i)]
                
		IF (gen_save.savaap) THEN BEGIN
		 FOR iaap=0,1 DO BEGIN
		  pia_load_iaap, phtaap(aapps(iaap)), phtiaap
            	  filename = phtiaap.admn+'.IAAP'
            	  save, phtiaap, FILE=!INTDIR+filename, /XDR
            	  WIDGET_LOG, wd_base,  $  
                    TEXT='Saving AAP measurement into '+!INTDIR+filename, $
                    ACTION='APPEND'
                 ENDFOR 
	         indcs = WHERE(phtcs.unit EQ STRMID(phtiaap.admn,4,8), count)
	         IF count GT 0 THEN BEGIN
		      PIA_LOAD_ICS, phtcs(indcs(0)), phtics
		      FILE='CS'+phtics.UNIT+'.ICS'
		      junk = FINDFILE(!intdir+FILE, COUNT=count)
		      IF count GT 0 THEN $
			MESSAGE, /INFO, $
		  	  'Compact Status file not written, already present' $
		      ELSE SAVE, phtics, FILE=!INTDIR+file, /XDR
	         ENDIF ELSE MESSAGE, /INFO, $
		    	'Compact Status not found in common, not saved' 
                ENDIF
	      ENDIF ELSE BEGIN
	        max_step = MAX(tmpscp.step)
		IF max_step GT 1 THEN $	
          	 WIDGET_LOG, wd_base, $ 
          	   TEXT =[' Chopped PHT-S dynamic calibration requested', $
		   	  ' but PHT-S measurement not subtracted, skipped' ], $
	             ACTION='APPEND'
	      ENDELSE
	  ENDIF
;
; Perform chopper frequency correction
;
	  IF (gen_correct.chfr) THEN BEGIN
	    INFO = 0 
            WIDGET_LOG, wd_base,TEXT =' Correcting signal losses due to chopping ',$
	             	ACTION='APPEND'
	    pat = fxpar(newheader, 'PRS_PATT', count=count)
	    IF count GT 0 THEN BEGIN
		INFO = ['Chopper loss corrections already automatically ',$
			'applied by "pattern reduction" SRD->SCP '] 
	    ENDIF ELSE IF slchop_proc THEN BEGIN	
		INFO = ['Chopper loss corrections are automatically ',$
			'applied by dynamic calibration for PHT-S chopped '] 
	    
	    ENDIF ELSE $ 
	    CHOP_FREQ_CORR, tmpscp, newheader=newheader, INFO = info

	    IF N_ELEMENTS(info) NE 0 THEN WIDGET_LOG, wd_base, TEXT =info, $ 	
	             ACTION='APPEND'	   
	  ENDIF 
	  
;
; replace scp meas within buffer by processed SCP
	  FILL_SCP, tmpscp, newheader, replace=scpps(i)  

; Write SCP data into PIA internal files

          IF (gen_save.savscp) THEN BEGIN
	    write_iscp, scpps(i), INFO = infosave
            WIDGET_LOG, wd_base, TEXT =infosave, ACTION='APPEND'
	    indcs = WHERE(phtcs.unit EQ STRMID(tmpscp.admn,4,8), count)
	    IF count GT 0 THEN BEGIN
		      PIA_LOAD_ICS, phtcs(indcs(0)), phtics
		      FILE='CS'+phtics.UNIT+'.ICS'
		      junk = FINDFILE(!intdir+FILE, COUNT=count)
		      IF count GT 0 THEN $
			MESSAGE, /INFO, $
		  	  'Compact Status file not written, already present' $
		      ELSE SAVE, phtics, FILE=!INTDIR+file, /XDR
	    ENDIF ELSE MESSAGE, /INFO, $
		    	'Compact Status not found in common, not saved' 
          ENDIF

          IF (gen_save.savscpf) THEN BEGIN
	    write_scp, tmpscp, newheader
	    text = ["Saving measurement " + dec_admn(tmpscp.admn), $
	          " as FITS file: " + !INTDIR + tmpscp.admn + '_SCP.FITS', ''] 
            WIDGET_LOG, wd_base, TEXT =text, $
             ACTION='APPEND' 
	  ENDIF
	  

   	ENDFOR         ; end loop for SCP

; Stop if just to SCP
	IF KEYWORD_SET(SCP) or KEYWORD_SET(PER_RPID) THEN GOTO, EXIT

; Stop if all measurements dyncalibrated or take those out from the list 
	IF N_ELEMENTS(scpps_slch) GT 0 THEN BEGIN
	   IF N_ELEMENTS(scpps_slch) EQ N_ELEMENTS(scpps) THEN GOTO, EXIT $
		ELSE BEGIN 
		  FOR i=0, N_ELEMENTS(scpps_slch)-1 DO BEGIN
		     ind = WHERE(scpps NE scpps_slch(i), nc)
		     scpps = scpps(ind)
		  ENDFOR
		ENDELSE 
	ENDIF
	
	firstspd = N_ELEMENTS(phtspd)

;;;;;;;; perform calibration ;;;;;;;;;;;;;;;;  

        	WIDGET_LOG, wd_base, TEXT =['','Beginning with calibration'], $
	             ACTION='APPEND'

NEWTDT:
if not KEYWORD_SET(auto_rec) THEN print, 'Warning: No AOT recognition'
	IF KEYWORD_SET(auto_rec) THEN BEGIN
	  handle_value, phtscp(scpps(0)).hdr, thdr
	  AOT = STRTRIM(FXPAR(thdr, 'EOHAAOTN'),2)
	  OBJECT = FXPAR(thdr, 'OBJECT')
	  TDTNR = STRMID(FXPAR(thdr, 'FILENAME'),4,8)
	  IF AOT EQ 'P37' or AOT EQ 'P17' THEN sparse = 1 ELSE $
	     sparse=0
	  text = ['','TDTNR: '+TDTNR + '  OBJECT: '+OBJECT+ '  AOT: ' + AOT] 
          WIDGET_LOG, wd_base, TEXT =text, $
	             ACTION='APPEND'
	ENDIF ELSE sparse=0

	IF KEYWORD_SET(sparse_map) OR sparse THEN nrtdt=6 ELSE nrtdt=8

	
	

; take first all measurements corresponding to one TDT and same detector:
;	tdt_scpps = scpps
	indeq = WHERE(STRMID(phtscp(scpps).admn,4,nrtdt) EQ $
			STRMID(phtscp(scpps(0)).admn,4,nrtdt) AND $
			phtscp(scpps).det EQ phtscp(scpps(0)).det, nele)
	ndel = nele
; flag all measurements corresponding to other TDTs or other dets 
	indneq = WHERE(STRMID(phtscp(scpps).admn,4,nrtdt) NE $
			STRMID(phtscp(scpps(0)).admn,4,nrtdt) OR $
			phtscp(scpps).det NE phtscp(scpps(0)).det, nnle)
NEWDET:
; take first all measurements corresponding to one detector 
;	indeq = WHERE(STRMID(phtscp(scpps).admn,4,nrtdt) EQ $
;			STRMID(phtscp(scpps(0)).admn,4,nrtdt), nele)
;	inddeq = WHERE(phtscp(scpps(indeq)).det EQ phtscp(scpps(0)).det, ndel)
; flag all measurements corresponding to other detectors 
;	indndeq = WHERE(phtscp(scpps(indeq)).det NE phtscp(scpps(0)).det,nndl)

	nndl = 0

; Check that the measurements are not there several times
	ndel_clean = REM_DUP(phtscp(scpps(indeq)).admn)
	IF N_ELEMENTS(ndel_clean) NE ndel THEN BEGIN
	   PRINT, 'Warning:  same measurement(s) several times present'
	   PRINT, 'Warning:  taking only the last position(s), cross fingers'
	   indeq = ndel_clean
	   ndel = N_ELEMENTS(ndel_clean)
	ENDIF


	IF KEYWORD_SET(auto_rec) THEN BEGIN
	  handle_value, phtscp(scpps(0)).hdr, thdr
	  AOT = STRTRIM(FXPAR(thdr, 'EOHAAOTN'),2)
	  IF AOT EQ 'P25' or AOT EQ 'P05' THEN abs_pht = 1 ELSE abs_pht=0
	ENDIF ELSE abs_pht=0

; special case: P05 / P25
        IF KEYWORD_SET(ABS_PHOT) or (abs_pht) THEN BEGIN
           nrmeas = ndel
	   detpix = pixel_det(phtscp(scpps(0)).det)
; only three measurements - either dark current or straylight
           IF nrmeas EQ 3 THEN BEGIN
	      darkmeas = 0
              get_wheels, phtscp(scpps(0)).admn, wheels
              IF STRMID(phtscp(scpps(0)).det, 0,1) EQ 'P' and wheels(0) NE 4 $
              OR STRMID(phtscp(scpps(0)).det, 0,1) EQ 'C' and wheels(0) NE 14 $
                 THEN darkmeas = 1
; Subtract dark current from source or straylight from FCS

; 1st case: dark - source - fcs

               IF darkmeas THEN BEGIN
                pia_load_iscp, phtscp(scpps(0)), darky
                pia_load_iscp, phtscp(scpps(1)), sourcy
                pia_load_iscp, phtscp(scpps(2)), fcsly
; source - dark
                SUBTR_darstr_MEAS, sourcy, darky, subt_meas, repl_index=scpps(1)
                 text = ['Dark current measurement detected, ', $
                         ' subtracting it from source measurement']
           	WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
; fcs - dark
                SUBTR_darstr_MEAS, fcsly, darky, subt_meas, repl_index=scpps(2)
                  text = ['No straylight measurement detected, ', $
                          ' subtracting dark meas. from FCS measurement']      
           	WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

; 2nd case: stray - source - fcs
               ENDIF ELSE BEGIN

                pia_load_iscp, phtscp(scpps(0)), strayly
                pia_load_iscp, phtscp(scpps(1)), sourcy
                pia_load_iscp, phtscp(scpps(2)), fcsly
                SUBTR_darstr_MEAS, fcsly, strayly, subt_meas, /STRAY,$
                	REPL_INDEX=scpps(2)
                  text = [ 'Straylight measurement detected, ', $ 
                           ' subtracting it from FCS measurement']                
           	WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

		newhdr = sourcy.hdr
                darkcur_orb, sourcy, newheader=newhdr, /SCP
		fill_scp, sourcy, newhdr, replace=scpps(1) 

                  text = ['No dark current measurement detected, ', $
                          ' subtracting default dark currents from source']
           	WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
                

               ENDELSE
              scpps = scpps(1:*)
              indneq = indneq - 1
              ndel = ndel - 1
; all 4 measurements - both dark current and straylight
           ENDIF ELSE IF nrmeas EQ 4 THEN BEGIN
              pia_load_iscp, phtscp(scpps(0)), strayly
              pia_load_iscp, phtscp(scpps(1)), darky
              pia_load_iscp, phtscp(scpps(2)), sourcy
              pia_load_iscp, phtscp(scpps(3)), fcsly
              SUBTR_darstr_MEAS, sourcy, darky, subt_meas, repl_index=scpps(2)
              SUBTR_darstr_MEAS, fcsly, strayly, subt_meas, /STRAY, $
              	repl_index=scpps(3)
              scpps = scpps(2:*)
              indneq = indneq - 2
              ndel = ndel - 2
                  text = ['Dark current measurement detected, ',     $ 
                          ' subtracting it from source measurement', $
                          'Straylight measurement detected, ' ,      $
                          ' subtracting it from FCS measurement']    
              WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
           ENDIF ELSE IF nrmeas EQ 2 THEN BEGIN
                pia_load_iscp, phtscp(scpps(0)), sourcy

		newhdr = sourcy.hdr
                darkcur_orb, sourcy, newheader=newhdr, /SCP
		fill_scp, sourcy, newhdr, replace=scpps(0) 

                  text = ['Only two measurements in P05/P25 detected, ', $
                          ' subtracting default dark currents from source']
           	WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

           ENDIF      
	ENDIF
; end of special case: P05 / P25
		                
	fcsmeas = 0
	FOR I=0,ndel-1 DO BEGIN
         	handle_value, phtscp(scpps(i)).fcsn, fcsn
		inda = WHERE(fcsn GT 0, nrfcs)
		IF nrfcs NE 0 THEN fcsmeas = [fcsmeas,scpps(i)]
	ENDFOR

; first case: no calibration measurement found

        IF N_ELEMENTS(fcsmeas) EQ 1 THEN BEGIN 

                text = ['No calibration measurement found', $
                         ' - using default responsivities']          

	   IF gen_respalt.single NE 1 THEN $ 
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

	   FOR I=0,ndel-1 DO BEGIN
		pia_load_iscp, phtscp(scpps(i)), tmpdata
		process_scp, tmpdata, tmpdata.hdr, 0, spdpos
	   ENDFOR
		
; second case: one calibration measurement found

	ENDIF ELSE IF N_ELEMENTS(fcsmeas) EQ 2 THEN BEGIN

	   pia_load_iscp, phtscp(fcsmeas(1)), tmpdata
	   inda = WHERE(tmpdata.fcsn EQ 0, nrfcs)

; -- absolute photometry case
	   IF nrfcs NE 0 THEN BEGIN
	   
                text = ['"Old" Absolute Photometry case found ', $
                        ' - processing accordingly ']
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
           
		process_scp, tmpdata, tmpdata.hdr, 2, spdpos  

	   ENDIF ELSE BEGIN

; -- normal photometry case

	    IF gen_respalt.single NE 1 THEN BEGIN
                text = ['One calibration measurement found', 	$  
                   'FCS meas. :'+dec_admn(phtscp(fcsmeas(1)).admn),       $
                   ' - calculating responsivities + calibrating with']        
                WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

		newheader = tmpdata.hdr
	        no_widget = (gen_respalt.single GE 1)
	        rchoice_batch = (gen_respalt.single-1)*2 - $ 
		    (gen_respalt.single GT 1)
		IF gen_respalt.single EQ 4 THEN BEGIN
			ind = WHERE(tmpdata.fcsn EQ 2, nind)
			IF nind EQ 0 THEN rchoice_batch = 1
			text = ["WARNING: NO FCS2 for calibration measurement ",$
			    "       "+ dec_admn(tmpdata.admn)+": using FCS1 "] 
			WIDGET_LOG, wd_base, $
			 text=text, ACTION='APPEND'
		ENDIF
			
		PIA_RESPONS, tmpdata, newheader, GROUP=group, $
			NO_WIDGET=no_widget, RCHOICE_BATCH = rchoice_batch, $
			OUT_OF_LIMITS=out_of_limits
		IF rchoice_batch LT 3 THEN out_of_limits=out_of_limits(0) $
		ELSE IF rchoice_batch LT 5 THEN out_of_limits=out_of_limits(1)
		
		IF out_of_limits THEN BEGIN
	          text= ['Default responsivity used because FCS el.power ',$
	                 'is out of limits. Process interactively if you  ',$
	                 'want nevertheless FCS derived responsivity   ']
            	  WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	          resp_method = 0
	        ENDIF ELSE resp_method=1
	    ENDIF ELSE BEGIN
                text = ['One calibration measurement found', 	$  
                   'FCS meas. :'+dec_admn(phtscp(fcsmeas(1)).admn),       $
                   ' but not used - Default response required ']        
                WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
		resp_method = 0
	    ENDELSE
	    
	    FOR I=0,ndel-1 DO BEGIN
			IF (scpps(I)) EQ fcsmeas(1) THEN GOTO, jump	
			pia_load_iscp, phtscp(scpps(I)), tmpdata
			newheader = tmpdata.hdr
			process_scp, tmpdata, newheader, resp_method, spdpos
		jump:
		
	    ENDFOR
		 
           ENDELSE

; third case: two calibration measurements found
; ---> interpolation

	ENDIF ELSE IF N_ELEMENTS(fcsmeas) EQ 3 THEN BEGIN
	 IF gen_respalt.single NE 1 THEN BEGIN 
	  IF gen_respalt.double EQ 0 THEN BEGIN 
           text = ['Two calibration measurements found',        $
                   '1st FCS meas. :'+dec_admn(phtscp(fcsmeas(1)).admn),    $
                   '2nd FCS meas. :'+dec_admn(phtscp(fcsmeas(2)).admn),    $     
                   ' - interpolating responses + calibrating with']
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
           
	   respons_interpol, fcsmeas(1), fcsmeas(2), resps, $
	    NO_WIDGET=gen_respalt.single, outlim=outlim
	   
	    
	   IF outlim THEN BEGIN
	     text= ['Default responsivity used because FCS el.power  ',$
	            'is out of limits. Process interactively if you  ',$
	            'want nevertheless FCS derived responsivity   ']
            WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	    resp_method = 0
	   ENDIF ELSE resp_method = 3

	  ENDIF ELSE IF gen_respalt.double EQ 1 THEN BEGIN
	   
           text = ['Two calibration measurements found',        $
                   '1st FCS meas. :'+dec_admn(phtscp(fcsmeas(1)).admn),    $
                   '2nd FCS meas. :'+dec_admn(phtscp(fcsmeas(2)).admn),    $     
                   ' averaging both + calibrating with ']
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	   respons_interpol, fcsmeas(1), fcsmeas(2), resps, $
	    NO_WIDGET=gen_respalt.single, outlim=outlim
           resps = 0
           
	   IF outlim THEN BEGIN
	     text= ['Default responsivity used because FCS el.power  ',$
	            'is out of limits. Process interactively if you  ',$
	            'want nevertheless FCS derived responsivity   ']
            WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	    resp_method = 0
	   ENDIF ELSE resp_method=1
           
           
          ENDIF ELSE BEGIN
           text = ['Two calibration measurements found',        $
                   '1st FCS meas. :'+dec_admn(phtscp(fcsmeas(1)).admn),    $
                   '2nd FCS meas. :'+dec_admn(phtscp(fcsmeas(2)).admn),    $     
                   ' but only second will be used ']
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

	   PIA_LOAD_ISCP, phtscp(fcsmeas(2)), tmpfcs
	   no_widget = (gen_respalt.single GE 1)
	   rchoice_batch = (gen_respalt.single-1)*2 - (gen_respalt.single GT 1)

	   IF gen_respalt.single EQ 4 THEN BEGIN
	   	   ind = WHERE(tmpdata.fcsn EQ 2, nind)
	   	   IF nind EQ 0 THEN rchoice_batch = 1
	   	   text = ["WARNING: NO FCS2 for calibration measurement ",$
	   	       "       "+ dec_admn(tmpdata.admn)+": using FCS1 "] 
	   	   WIDGET_LOG, wd_base, $
	   	    text=text, ACTION='APPEND'
	   ENDIF
	   
	   PIA_RESPONS, tmpfcs, newheader, GROUP=group, $
			NO_WIDGET=no_widget, RCHOICE_BATCH =rchoice_batch, $
			out_of_limits=out_of_limits
                    
	   resps = 0

	   IF out_of_limits THEN BEGIN
	     text= ['Default responsivity used because FCS el.power  ',$
	            'is out of limits. Process interactively if you  ',$
	            'want nevertheless FCS derived responsivity   ']
            WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	    resp_method = 0
	   ENDIF ELSE resp_method=1
	   
          ENDELSE   
	 ENDIF ELSE BEGIN
           text = ['Two calibration measurements found',        $
                   '1st FCS meas. :'+dec_admn(phtscp(fcsmeas(1)).admn),    $
                   '2nd FCS meas. :'+dec_admn(phtscp(fcsmeas(2)).admn),    $     
                   ' none of them used - Default responsivity required ']
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	   resps = 0
	   resp_method = 0
	 ENDELSE
	   
	  	 
	   FOR I=0,ndel-1 DO BEGIN
			IF (scpps(I)) EQ fcsmeas(1) THEN GOTO, jump2	
			IF (scpps(I)) EQ fcsmeas(2) THEN GOTO, jump2	
			pia_load_iscp, phtscp(scpps(I)), tmpdata
			
			process_scp, tmpdata, tmpdata.hdr, resp_method, spdpos, $
				respint=resps
		jump2:
		
	   ENDFOR

; fourth case: more than two calibration measurements found - only in case
; of multi-filter mapping nominally possible
; ---> interpolation in pairs

	ENDIF ELSE IF N_ELEMENTS(fcsmeas) GT 3 THEN BEGIN
	 IF gen_respalt.single NE 1 THEN BEGIN
	   tfcs=[' ']
	   FOR Ifcsmeas=1,N_ELEMENTS(fcsmeas)-1 DO $
	     tfcs=[tfcs,'FCS meas. :'+dec_admn(phtscp(fcsmeas(ifcsmeas)).admn)]
           text = ['More than 2 calibration measurements found',  tfcs, $
                   ' - trying my best using them pairwise ',      $
                   ' - interpolating responses + calibrating with']	   
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'

		FOR ipair=0,(N_ELEMENTS(fcsmeas)-1) / 2 - 1 DO BEGIN
		   fcs1 = fcsmeas(1+ipair*2)
		   fcs2 = fcsmeas(2+ipair*2)
		   
	   	   respons_interpol, fcs1, fcs2, resps, $
	   	     NO_WIDGET=gen_respalt.single

	   	   ind = WHERE(scpps GT fcs1 AND scpps LT fcs2, nind)
	   	   IF nind EQ 0 THEN BEGIN
	   	     PRINT, 'No meas. found within 2 consecutive FCSs'
		     PRINT, 'Trying to continue '
		   ENDIF ELSE BEGIN
	   	     FOR I=0,N_ELEMENTS(ind)-1 DO BEGIN
			pia_load_iscp, phtscp(scpps(ind(i))), tmpdata
			process_scp, tmpdata, tmpdata.hdr, 3, spdpos, $
				respint=resps
		     ENDFOR
		   ENDELSE
		 ENDFOR

	ENDIF ELSE BEGIN
           text = ['More than two calibration measurements found',        $
                   ' none of them used - Default responsivity required ']
           WIDGET_LOG, wd_base,  TEXT= text, ACTION='APPEND'
	   resps = 0
	   resp_method = 0

	   FOR ipair=0,(N_ELEMENTS(fcsmeas)-1) / 2 - 1 DO BEGIN
		   fcs1 = fcsmeas(1+ipair*2)
		   fcs2 = fcsmeas(2+ipair*2)

	   	   ind = WHERE(scpps GT fcs1 AND scpps LT fcs2, nind)
	   	   IF nind EQ 0 THEN BEGIN
	   	     PRINT, 'No meas. found within 2 consecutive FCSs'
		     PRINT, 'Trying to continue '
		   ENDIF ELSE BEGIN
	   	     FOR I=0,N_ELEMENTS(ind)-1 DO BEGIN
			pia_load_iscp, phtscp(scpps(ind(i))), tmpdata
			process_scp, tmpdata, tmpdata.hdr, resp_method, spdpos, $
				respint=resps
		     ENDFOR
		   ENDELSE
	    ENDFOR

	 ENDELSE
	ENDIF


; check if there is data corresponding to another detector 
	IF nndl NE 0 THEN BEGIN
	 PRINT, 'Something is WRONG !!!!, I can not get here'
           WIDGET_LOG, wd_base,  $  
              TEXT='Wow - more data coming from another detector', $
              ACTION='APPEND'
		scpps = scpps(indndeq(0):*)
		GOTO, NEWDET
	ENDIF

; check if there is data corresponding to another TDT 
	IF nnle NE 0 THEN BEGIN
           WIDGET_LOG, wd_base,  $  
              TEXT='Wow - more data coming ...', $
              ACTION='APPEND'
		scpps = scpps(indneq(0):*)
		GOTO, NEWTDT
	ENDIF


	lastspd = N_ELEMENTS(phtspd) - 1
	
	nspd = lastspd - firstspd 

	save_spdaap = gen_save.savspd + gen_save.savaap

	info = {admn:'', filter:'', apert:'', polar:'', pixel:0, chops:0, $
		drift_ind:0, resp:0.,flux:0.,flunc:0.,bright:0.,brunc:0.}

	infotmp = info
	

	FOR I=0,nspd DO BEGIN
		pia_load_ispd, phtspd(firstspd+i), tmpdata
		add_coordinates, tmpdata, INFO = infoaddc
		IF N_ELEMENTS(infoaddc) NE 0 THEN $ 
                  WIDGET_LOG, wd_base,  TEXT= '  '+ infoaddc, ACTION='APPEND'
		   
		process_spd, tmpdata, tmpdata.hdr, tmpaar, aapps
		
		IF gen_correct.spoint AND STRMID(tmpaar.det,0,1) EQ 'S' THEN $
		  IF N_ELEMENTS(tmpaar.ra) EQ 1 THEN BEGIN
		    
        	    newhdr = tmpaar.hdr
		    ra_int = FXPAR(newhdr,'ATTRA')
        	    dec_int = FXPAR(newhdr,'ATTDEC')
		    
		    diff_radec_yz, tmpaar.ra, tmpaar.dec, tmpaar.roll, $
			ra_int, dec_int, dy_off, dz_off
; correct only if above limit 
		    limit = 0. 
		    distance = SQRT(dy_off*dy_off + dz_off*dz_off)
		    IF distance GT limit THEN BEGIN

                     correct_s_offset, tmpaar, newhdr, dy_off, dz_off, $
                        ERROR=error, off_cor=off_cor
          	     WIDGET_LOG, wd_base,  $  
              	      TEXT=' Correcting '+ $
		   	dec_admn(tmpaar.admn) + ': PHT-S pointing offset', $
              		ACTION='APPEND'
		    
		     IF error EQ 0 THEN $
		       fill_aap, tmpaar, newhdr, replace=aapps
		    ENDIF
		ENDIF
	
; fill info - BEGIN
		IF KEYWORD_SET(summ_info) THEN BEGIN

		GIVE_CS, tmpdata, cst, in, conv_cst
		npix = N_ELEMENTS(tmpdata.mnpw(*,0))
		nrpcp = N_ELEMENTS(tmpdata.mnpw(0,*))
		ddd = STRMID(tmpdata.admn, 0,1)
		res = fxpar(tmpdata.hdr, 'PRC_R*')
		resu = fxpar(tmpdata.hdr, 'PRS_DINT', count=nc)
		drift_ind = fltarr(npix) - 1.

		IF nc EQ 1 THEN BEGIN
		 ad = WHERE(phtdrift.admn EQ tmpdata.admn, nnc)
		 IF nnc GT 1 THEN ad=max(ad) ELSE IF nnc EQ 1 THEN ad=ad(0) $
		   ELSE GOTO, jump_out
		 handle_value, phtdrift(ad).stab, stab 
		 drift_ind = stab.stabflag(0:npix-1,0)
jump_out:
		ENDIF		 

		FOR ipix = 0,npix-1 DO BEGIN
		 decode_admn, tmpdata.admn, dec, decstr_sh=decstr_sh
		 infotmp.admn = decstr_sh
		 IF ddd EQ 'C' THEN BEGIN
		  infotmp.filter = conv_cst.c2ps(in) 
		  infotmp.polar = STRMID(conv_cst.c1ps(in),4,8)
		 ENDIF ELSE IF ddd EQ 'P' THEN BEGIN
		  infotmp.filter = conv_cst.c3ps(in)
		  infotmp.apert = conv_cst.c2ps(in)
		  infotmp.polar = STRMID(conv_cst.c1ps(in),4,8)
		 ENDIF 
		 infotmp.pixel = ipix+1
		 infotmp.chops = nrpcp
		 infotmp.resp = res(ipix)
		 infotmp.drift_ind = drift_ind(ipix)
		 infotmp.flux = tmpaar.mnfl(ipix,0)
		 infotmp.flunc = tmpaar.mnfu(ipix,0)
		 infotmp.bright = tmpaar.mnbr(ipix,0)
		 infotmp.brunc = tmpaar.mnbu(ipix,0)
		 info = [info, infotmp]
		ENDFOR
; fill info - END
		    
		ENDIF
		
		IF save_spdaap GT 0 THEN BEGIN
                 IF (gen_save.savspd) THEN BEGIN
            	  filename = tmpdata.admn+'.ISPD'
            	  phtispd = tmpdata
            	  save, phtispd, FILE=!INTDIR+filename, /XDR
            	  WIDGET_LOG, wd_base,  $  
                    TEXT='Saving SPD measurement into '+!INTDIR+filename, $
                    ACTION='APPEND'
                 ENDIF
                 IF (gen_save.savaap) THEN BEGIN
            	  filename = tmpaar.admn+'.IAAP'
            	  phtiaap = tmpaar
            	  save, phtiaap, FILE=!INTDIR+filename, /XDR
            	  WIDGET_LOG, wd_base,  $  
                    TEXT='Saving AAP measurement into '+!INTDIR+filename, $
                    ACTION='APPEND'
                 ENDIF
	         indcs = WHERE(phtcs.unit EQ STRMID(tmpdata.admn,4,8), count)
	         IF count GT 0 THEN BEGIN
		      PIA_LOAD_ICS, phtcs(indcs(0)), phtics
		      FILE='CS'+phtics.UNIT+'.ICS'
		      junk = FINDFILE(!intdir+FILE, COUNT=count)
		      IF count GT 0 THEN $
			MESSAGE, /INFO, $
		  	  'Compact Status file not written, already present' $
		      ELSE SAVE, phtics, FILE=!INTDIR+file, /XDR
	          ENDIF ELSE MESSAGE, /INFO, $
		    	'Compact Status not found in common, not saved' 
                ENDIF

              IF (gen_save.savspdf) THEN BEGIN
	        write_spd, tmpdata, tmpdata.hdr
	        text = ["Saving measurement " + dec_admn(tmpdata.admn), $
	            " as FITS file: " + !INTDIR + tmpdata.admn + '_SPD.FITS',''] 
                WIDGET_LOG, wd_base, TEXT =text, $
                ACTION='APPEND' 
	      ENDIF
	      
	      IF (gen_save.savaapf) THEN BEGIN
	        filename=!INTDIR+tmpaar.admn + '_AAP.FITS'

	        text = ["Saving measurement " + dec_admn(tmpaar.admn), $
	            " as FITS file: " + filename,''] 
                WIDGET_LOG, wd_base, TEXT =text, $
                ACTION='APPEND' 

		fill_aap, tmpaar, tmpaar.hdr
		posi = N_ELEMENTS(phtaap) - 1
		mask = [0,intarr(17) + 1]
		write_aap_multi, posi, accvector=mask, filename=filename
		aap_init, index=posi
	      ENDIF
	       	
		
		
	ENDFOR

; print info 
	IF KEYWORD_SET(summ_info) THEN BEGIN
		info = info(1:*)
		TEXT = ['Photometric summary table: ', $
		        '===========================', '', $
'Note: This table is actually designed for pointing staring observations,', $
'if the number of raster points / chopper plateaux (RC) > 1 then only', $
'the info corresponding to first RP/Ch.plateau will be quoted', '', $
'Measurement    Filter    Apert    Polar   Pix RC StFL Resp |  Flux   +/-     |   Br.  +/-       |']


		text = [text, $
STRING(format="(A14,1x,A9,1x,A7,1x,A7,1x,I3,1x,I3,1x,I2,1x,F6.2,1x,E9.2,1x,E8.1,1x,E8.1,1x,E8.1)",info)]
		
		PIA_DISP_PRINT_FILE, '', TITLE='Photometric summary table',$
			Text=text
	ENDIF
exit:
        WIDGET_LOG, wd_base,  $ 
              TEXT =['','Ready READY Ready READY Ready'], $
               ACTION='APPEND'

        PRINT, STRING(7B)
        WAIT,1
        PRINT, STRING(7B)

;	Xmanager,No_BLOCK=1, "WIDGET_LOG", wd_base, GROUP=group
	
		    
	RETURN
	END

