pro mem2,blur,psf,iter,lambda,fftp,init=init,niter=niter, $
	scale=scale,sigma=sigma,chimin=chimin,stall=stall, $
	display=display,region=region,header=header
;+
;			mem2
;
; Computes the maximum likelyhood deconvolution.
;
; CALLING SEQUENCE:
;	mem2,blur,psf,iter,fftp,lambda,/init,niter=N,
;		scale=R,sigma=R,chimin=R,stall=R,display=S,region=V,
;		header=S
; INPUTS:
;	blur - unrestored raw image
;	psf - point spread function (normalized to total = 1.0)
;
; INPUTS/OUTPUTS:
;	iter - restored image.  if undefined, set to a scalar, or keyword INIT
;		is specified it will be initialized with all pixels having the
;		average flux value of BLUR
;
; OPTIONAL INPUTS/OUTPUTS:
;
;	lambda - internal array used by the routine.  If you want to continue
;		iterating in a subsequent call you must supply this array in
;		the calling sequence and supply it in subsequent calls.
;		If keyword /INIT is specified, lambda is undefined or lambda
;		is set to a scalar, MEM2 initializes lambda (0th iteration).
;
;	fftp - the fft of the psf.  If not defined or if INIT is specified then
;		then it will be computed and returned as an output.  It
;		can be passed to subsequent calls to save computing time.
;
; OPTIONAL KEYWORD PARAMETER INPUTS:
;
;	/INIT - specifies that X and FFTP should be initialized. This should
;		only be used on the first call.
;	NITER=n - specifies the maximum number of iterations.
;		Default = 50.  (on output it gives the actual number of
;		iterations completed).
;	SCALE=r - specifies the number of Poisson counts per flux unit.
;		This is used by the Chi-sqaure termination test.  If set
;		to 0.0 then only white noise is assumed. (default = 1.0)
;	SIGMA=r - standard deviation of the white noise.  This is used by
;		the Chi-squared termination test. (default = 0.0)
;	CHIMIN=r - Chi-squared value required for termination.  Default=1.0
;		If set to 0.0 this test is disabled.
;	STALL=r - If the chi-squared changes by less than STALL, the iterations
;		are assumed to be stalled and processing is terminated.
;		(default = 0.001).  If set to 0.0 this test is disabled.
;	DISPLAY - IDL command to display the ITER.
;		examples:
;			 DISPLAY = 'tvscl,alog(iter>0.01<100.0)'
;			 DISPLAY = 'contour,iter>0'
;		if not supplied then the iteration is not displayed
;	REGION - region of image to use for the chi=squared test given as
;		a four element vector, [starting sample, ending sample,
;		starting line, ending line].  Default is to use the entire
;		image.
;	HEADER - input/output FITS header.  If supplied it will be updated
;		with restoration history.
; METHOD:
;	From Hollis, and Dorband, "Comparing Restored and VLA imagery of
;	R. Aquarii"  AJ (1992 Feb. 10).
;
;	Let:
;		b = unrestored raw input image
;		p = point spread function
;		lambda = lambda's from the previous iteration.  They
;			are all set to 0.0 for the 0th iteration
;		C(q,t) = the convolution of q and t
;		pr = 180 degree rotation of p
;	The next iteration is computed by:
;		d = exp(C(lambda,pr)
;		d = d/total(d)
;		x = C(d,p)
;		lambda = lambda + alog(b/x/total(b))	;updated lambda's
;		x = x*total(b)				;resulting image
;
;	All convolutions are computed with FFT's
;
; EXAMPLES:
;
;	Do up to 50 iterations with standard termination tests for
;	Poison statistics:
;
;		mem2,blur,psf,result
;
;	Do 80 iterations with no tests for termination.
;	Follow it with 40 more.
;
;		mem2,blur,psf,result,lambda,chimin=0,stall=0,niter=80,/init
;		mem2,blur,psf,result,lambda,chimin=0,stall=0,niter=40
;
;	Use both Poisson noise (where 1 flux unit = 7.5 counts)
;	and additive Gaussian noise with a sigma of 2.0 for testing
;	convergence.  Results are displayed on the
;	image display after each iteration.
;
;		mem2,blur,psf,result,scale=7.5,sigma=2.0, $
;				display='tvscl,iter<100'
;
;-
;-------------------------------------------------------------------------------
	if n_params(0) lt 1 then begin
	    print,'CALLING SEQUENCE: mem2,blur,psf,iter,[lambda,fftp]'
	    print,'Optional keyword inputs: init,niter,scale,sigma,chimin,'
	    print,'                         stall,display,region,header'
	    return
	end
;
; set default parameters
;
	if n_elements(init) eq 0 then init = 0
	if n_elements(niter) eq 0 then niter = 50
	if n_elements(scale) eq 0 then scale = 1.0
	if n_elements(sigma) eq 0 then sigma = 0.0
	if n_elements(chimin) eq 0 then chimin = 1.0
	if n_elements(stall) eq 0 then stall = 0.001
	if n_elements(display) eq 0 then display = ''
;
; compute FFT of the PSF if required.
;
	if (init ne 0) or (n_elements(fftp) lt 2) then $
			psf_fft,blur,psf,fftp
	cfftp = conj(fftp)			;conjugate of fftp
;
; initialize lambdas if required:
;
	s = size(blur) & ns=s(1) & nl=s(2) & npixels = ns*nl
	if (init ne 0) or (n_elements(lambda) le 1) then lambda = fltarr(ns,nl)
	totb = total(blur)
;
; compute chisq for previous iteration
;
	iter = exp(float( fft(fft(lambda,-1)*cfftp,1) )*npixels)
	iter = iter*(totb/total(iter))
	reblur = float(fft(fft(iter,-1)*fftp,1))*npixels
	compute_chisq,blur,reblur,sigma,scale,chisqin,region=region
	print,' Initial CHISQ per degree of freedom = ',chisqin
;
; loop on iterations
;
	for i = 1,niter do begin
		ratio = (blur>1e-10)/(reblur>1e-10)
		lambda = lambda + alog(ratio)
;
; compute new image
;
		iter = exp(float(fft(fft(lambda,-1)*cfftp,1))*npixels)
		iter = iter*(totb/total(iter))
		reblur = float(fft(fft(iter,-1)*fftp,1))*npixels

;
; display results
;
		if display ne '' then istat = execute(display)
;
; compute chisq
;
		compute_chisq,blur,reblur,sigma,scale,chisq,region=region
		print,'Iteration '+strtrim(i,2)+ $
			' CHISQ per degree of freedom = ',chisq
;
; should we terminate
;
		if chisq le chimin then begin
			message = 'Chi-square value achieved'
			mes_value = 0
			goto,done
		end

		if abs(chisq-chisqin) lt stall then begin
			message='Change in chi-square less then '+ $
				strtrim(stall,2)
			mes_value = 1
			goto,done
		end
		chisqin = chisq
	end; for i
done:
	niter = i<niter
	if n_elements(message) lt 1 then BEGIN
			 message = 'Max. number of iterations = '+ $
			 strtrim(niter,2)+' reached'
		mes_value = 2
	ENDIF
	mess2 = 'TERMINATION: '+message
	print,mess2
	if n_elements(header) lt 1 then return
; pass info as keywords (CG)
	fxaddpar, header, 'PRM_DCME', 'MEM2', $
	 ' Image deconvolution method used', BEFORE='XTENSION'
	fxaddpar, header, 'PRM_DCIT', niter, $
	 ' MEM2: number of performed iterations', BEFORE='XTENSION'
	fxaddpar, header, 'PRM_DCCH', chisq, $
	 ' MEM2: Final Chi-square achieved', BEFORE='XTENSION'
	fxaddpar, header, 'PRM_DCSG', sigma, $
	 ' MEM2: Map intrinsic noise used', BEFORE='XTENSION'
	fxaddpar, header, 'PRM_DCTM', mes_value, $
	 ' MEM2: '+ message, BEFORE='XTENSION' 
;	sxaddhist,'MEM2: '+strtrim(niter,2)+' iterations',header
;	sxaddhist,'Chi-squared = '+strtrim(chisq,2)+' for SCALE='+ $
;		strtrim(scale,2)+'  SIGMA='+strtrim(sigma,2),header
;	sxaddhist,message,header
return
end
