pro extast,hdr,astr,noparams
;+
; NAME:
;	EXTAST
; PURPOSE:
;	Extract astrometry parameters from a FITS image header.
;	The astrometry in the header can be in either CD (Coordinate
;	description) format, or CROTA and CDELT (AIPS-type) format.
;	However, the output astrometry will always be in CD format.
;
; CALLING SEQUENCE:
;	EXTAST, hdr, [ astr, noparams ]   
;
; INPUT:
;	HDR - variable containing the FITS header (string array)
;
; OUTPUTS:
;	ASTR - Anonymous structure containing astrometry info from the FITS 
;		header.  ASTR always contains the following tags (even though 
;		some projections do not require all the parameters)
;   	 .CD   -  2 x 2 array containing the astrometry parameters CD1_1 CD1_2
;		in DEGREES/PIXEL                                   CD2_1 CD2_2
;	 .CDELT - 2 element vector giving physical increment at reference pixel
;	 .CRPIX - 2 element vector giving X and Y coordinates of reference pixel
;		(def = NAXIS/2)
;	 .CRVAL - 2 element double precision vector giving R.A. and DEC of 
;		reference pixel in DEGREES
;	 .CTYPE - 2 element string vector giving projection types, default
;		['RA---TAN','DEC--TAN']
;	 .LONGPOLE - scalar longitude of north pole (default = 180) 
;        .PROJP1 - Scalar parameter needed in some projections
;	 .PROJP2 - Scalar parameter needed in some projections
;
;	NOPARAMS -  Scalar indicating the results of EXTAST
;		-1 = Failure - Header missing astrometry parameters
;		0 = Success - Header contains CD00n00m + CDELT* astrometry
;		1 = Success - Header contains CROTA + CDELT (AIPS-type) astrometry
;		2 = Success - Header contains CDn_m astrometry
;
; PROCEDURE
;	EXTAST checks for astrometry parameters in the following order:
;	(1) the CD matrix CD1_1,CD1_2... plus CRPIX and CRVAL.   
;	(2) the CD matrix CD001001,CD001002...plus CRPIX and CRVAL
;	(3) CROTA2 (or CROTA1) and CDELT plus CRPIX and CRVAL.
;	See the Memo: Representations of Celestial Coordinates in FITS by
;		Griesen and Calabretta, available at fits.cv.nrao.edu
;
; NOTES:
;	(1) An anonymous structure is created to avoid structure definition
;		conflicts.    This is needed because some projection systems
;		require additional dimensions (i.e. spherical cube
;		projections require a specification of the cube face).
;
; PROCEDURES CALLED:
;	ZPARCHECK, SXPAR
; REVISION HISTORY
;	Written by B. Boothman 4/15/86
;	Accept CD001001 keywords               1-3-88
;	Accept CD1_1, CD2_1... keywords    W. Landsman    Nov. 92
;-
; On_error,2

 if ( N_params() LT 2 ) then begin
     print,'Syntax - extast, hdr, astr, [ noparams ]'
     return
 endif

 radeg = 180.0D0/!DPI

 zparcheck,'EXTAST',hdr,1,7,1,'FITS image header'   ;Make sure valid header
 noparams = -1                                    ;Assume no astrometry to start


  crval = sxpar( hdr, 'CRVAL*', Count = N )
     if N LT 2 then return 		;No CRVAL parameters

  crpix = sxpar( hdr, 'CRPIX*', Count = N )
     if N lt 2 then return                 ;No CRPIX parameters?


 ctype = sxpar( hdr, 'CTYPE*', Count = N_ctype)
 if N_ctype EQ 0 then ctype = ['RA---TAN','DEC--TAN']

 CD11 = sxpar( hdr, 'CD001001', COUNT = N_CD001 )

 if N_CD001 EQ 1 then begin 
        CD12 = sxpar( hdr, 'CD001002')
        CD21 = sxpar( hdr, 'CD002001')  
        CD22 = sxpar( hdr, 'CD002002')

; Get CDELT values.   If no CDELT keywords found, then assume that they
; are incorporated in the CD00n00m values, and convert to CDn_m

        CDELT = sxpar( hdr, 'CDELT*', Count = N )
        if N lt 2 then begin  
		fits_cd_fix, hdr, /REV
		cdelt = [1.0D, 1.0D ]
		noparams = 2
	endif else noparams = 0

 endif else begin
    CD11 = sxpar( hdr,'CD1_1', Count = N_CD1 )
    if N_CD1 EQ 1 then begin	    ;If CD parameters don't exist, try CROTA
	CD12 = sxpar( hdr, 'CD1_2' )
	CD21 = sxpar( hdr, 'CD2_1' )  
	CD22 = sxpar( hdr, 'CD2_2' )
	cdelt = [1.0D, 1.0D ]
	noparams = 2
    endif else begin

; Now get rotation, first try CROTA2, if not found try CROTA1, if that
; not found assume North-up.   Then convert to CD matrix

        CDELT = sxpar( hdr, 'CDELT*', Count = N )
        if N lt 2 then return       ;Must have CDELT1 and CDELT2

        CROTA = sxpar( hdr,'CROTA2', Count = N)  
	if N EQ 0 then  crota = sxpar(hdr, 'CROTA1', COUNT = N )
        if N EQ 0 then crota = 0.0
	crota = crota /  RADEG	     
	CD11 = cos(crota)
	CD12 = sin(crota)
	CD21 = -sin(crota)
	CD22 = cos(crota)
        noparams = 1           ;Signal AIPS-type astrometry found
     

    endelse

  endelse

  cd = [ [ cd11, cd21 ] , [ cd12, cd22] ]    ;Error in array order corrected
 
  if N_elements( cdelt ) LT 2 then begin
	cdelt = sxpar( hdr, 'CDELT*', Count = N)
	if N LT 2 then cdelt = [1.0D,1.0D]
  endif 

  projection = strmid( ctype(0), 5, 3)
  if (projection EQ 'TAN') then begin
	longpole = 180
	projp1 = -1.
	projp2 = -2.
   endif else begin
	longpole = sxpar( hdr, 'LONGPOLE')
	projp1 =  sxpar( hdr,'PROJP1')
	projp2 = sxpar( hdr,'PROJP2')
  endelse

; Note that the dimensions and datatype of each tag must be explicit, so that
; there is no conflict with structure definitions from different FITS headers

  ASTR = {CD: double(cd), CDELT: double(cdelt), $
		CRPIX: float(crpix), CRVAL:double(crval), $
		CTYPE: string(ctype), LONGPOLE: float( longpole(0)),  $
		PROJP1: float(projp1(0)), PROJP2: float(projp2(0))}

  return
  end
